<?php
session_start();
include 'conexao.php';

if (!isset($_SESSION['aluno'])) {
    header("Location: login_aluno.php");
    exit;
}

$aluno = $_SESSION['aluno'];
$aluno_id = $aluno['id'];

// --- CÓDIGO DE VERIFICAÇÃO DE PAGAMENTO AJUSTADO ---
date_default_timezone_set('Africa/Luanda');

// Mapeia os nomes dos meses em inglês para português para a verificação
$meses_pt = [
    'January' => 'Janeiro', 'February' => 'Fevereiro', 'March' => 'Março',
    'April' => 'Abril', 'May' => 'Maio', 'June' => 'Junho',
    'July' => 'Julho', 'August' => 'Agosto', 'September' => 'Setembro',
    'October' => 'Outubro', 'November' => 'Novembro', 'December' => 'Dezembro'
];
$mes_ingles_atual = date('F'); // Nome do mês atual em inglês
$mes_atual = $meses_pt[$mes_ingles_atual] ?? 'Mês inválido'; // Converte para português
$ano_atual = date('Y'); // Ano atual

// A consulta agora usa o nome do mês em português e a coluna 'mes'
$sql_check_pagamento = "
    SELECT id FROM propinas 
    WHERE aluno_id = ? AND mes = ? AND ano = ?
    LIMIT 1
";
$stmt_check = $conn->prepare($sql_check_pagamento);
$stmt_check->bind_param("isi", $aluno_id, $mes_atual, $ano_atual);
$stmt_check->execute();
$res_check = $stmt_check->get_result();
$propina_paga_mes_atual = ($res_check->num_rows > 0);
$stmt_check->close();

// Se a propina não estiver paga, para a execução do código e exibe a mensagem de erro
if (!$propina_paga_mes_atual) {
    echo '
    <!DOCTYPE html>
    <html lang="pt">
    <head>
        <meta charset="UTF-8">
        <title>Acesso Restrito</title>
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
        <style>
            body { font-family: \'Segoe UI\', sans-serif; background: #f4f6f9; margin: 0; padding: 20px; text-align: center; }
            .container { max-width: 600px; margin: 50px auto; padding: 20px; background: #fff; border-radius: 8px; box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1); }
            .voltar { margin-top: 20px; }
            .voltar a { text-decoration: none; color: white; background: #003366; padding: 10px 20px; border-radius: 5px; }
        </style>
    </head>
    <body>
        <div class="container">
            <h2>Acesso Restrito</h2>
            <p>O boletim escolar não pode ser exibido. Por favor, regularize o pagamento da propina do mês de <strong>'. htmlspecialchars(ucfirst($mes_atual)) .'</strong>.</p>
            <div class="voltar">
                <a href="painel_aluno.php"><i class="fas fa-arrow-left"></i> Voltar ao Painel</a>
            </div>
        </div>
    </body>
    </html>
    ';
    exit;
}

// --- FIM DO CÓDIGO DE VERIFICAÇÃO ---

// Trimestre selecionado (padrão: 1º)
$trimestreSelecionado = isset($_POST['trimestre']) ? intval($_POST['trimestre']) : 1;

$stmt = $conn->prepare("
    SELECT
        d.nome AS disciplina,
        n.mac,
        n.npp,
        n.npt
    FROM notas n
    JOIN disciplinas d ON n.disciplina_id = d.id
    WHERE n.aluno_id = ? AND n.trimestre = ?
    ORDER BY d.nome
");
$stmt->bind_param("ii", $aluno_id, $trimestreSelecionado);
$stmt->execute();
$result = $stmt->get_result();

$notas = $result->fetch_all(MYSQLI_ASSOC);

function formatarNota($nota) {
    if (!is_numeric($nota) || $nota <= 0) {
        return '-';
    }
    return number_format($nota, 2);
}

function getCorNota($nota) {
    if (is_numeric($nota) && $nota > 0 && $nota < 10) {
        return 'vermelho';
    }
    return '';
}
?>
<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <title>Boletim do Aluno</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        body {
            font-family: 'Segoe UI', sans-serif;
            background: #f4f6f9;
            margin: 0;
            padding: 20px;
        }
        h2 {
            text-align: center;
            color: #003366;
        }
        form {
            text-align: center;
            margin-bottom: 20px;
        }
        select {
            padding: 6px 10px;
            font-size: 16px;
            border-radius: 5px;
        }
        .tabela-boletim {
            margin: 10px auto;
            width: 95%;
            border-collapse: collapse;
            background: white;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .tabela-boletim th, .tabela-boletim td {
            padding: 10px;
            border: 1px solid #ccc;
            text-align: center;
        }
        .tabela-boletim th {
            background-color: #003366;
            color: white;
        }
        .vermelho {
            color: red;
            font-weight: bold;
        }
        .voltar {
            display: block;
            margin: 20px auto;
            text-align: center;
        }
        .voltar a {
            text-decoration: none;
            color: white;
            background: #003366;
            padding: 10px 20px;
            border-radius: 5px;
        }
        .voltar a:hover {
            background: #0056b3;
        }
    </style>
</head>
<body>

<h2>📋 Boletim Escolar de <?= htmlspecialchars($aluno['nome']) ?></h2>

<form method="post">
    <label for="trimestre"><strong>Selecionar Trimestre:</strong></label>
    <select name="trimestre" id="trimestre" onchange="this.form.submit()">
        <option value="1" <?= ($trimestreSelecionado == 1 ? 'selected' : '') ?>>1º Trimestre</option>
        <option value="2" <?= ($trimestreSelecionado == 2 ? 'selected' : '') ?>>2º Trimestre</option>
        <option value="3" <?= ($trimestreSelecionado == 3 ? 'selected' : '') ?>>3º Trimestre</option>
    </select>
</form>

<?php if (empty($notas)): ?>
    <p style="text-align:center;">⚠️ Nenhuma nota lançada para o trimestre selecionado.</p>
<?php else: ?>
    <table class="tabela-boletim">
        <thead>
            <tr>
                <th>Disciplina</th>
                <th>MAC</th>
                <th>NPP</th>
                <th>NPT</th>
                <th>MT</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($notas as $nota): ?>
                <?php
                    $mac = (float)($nota['mac'] ?? 0);
                    $npp = (float)($nota['npp'] ?? 0);
                    $npt = (float)($nota['npt'] ?? 0);
                    
                    $mt = 0;
                    if ($mac > 0 && $npp > 0 && $npt > 0) {
                        $mt = ($mac + $npp + $npt) / 3;
                    }
                ?>
                <tr>
                    <td><?= htmlspecialchars($nota['disciplina']) ?></td>
                    <td class="<?= getCorNota($mac) ?>"><?= formatarNota($mac) ?></td>
                    <td class="<?= getCorNota($npp) ?>"><?= formatarNota($npp) ?></td>
                    <td class="<?= getCorNota($npt) ?>"><?= formatarNota($npt) ?></td>
                    <td class="<?= getCorNota($mt) ?>"><strong><?= formatarNota($mt) ?></strong></td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
<?php endif; ?>

<div class="voltar">
    <a href="painel_aluno.php"><i class="fas fa-arrow-left"></i> Voltar ao Painel</a>
</div>

</body>
</html>